/************************************************************************/
/*									*/
/*			N I O S . C					*/
/*			-----------					*/
/*									*/
/*	Copyright (C) 1988 Digital Research Inc. All rights		*/
/*	reserved. The Software Code contained in this listing is	*/
/*	proprietary to Digital Research Inc., Monterey,			*/
/*	California, and is covered by U.S. and other copyright		*/
/*	protection. Unauthorized copying, adaption, distribution,	*/
/*	use or display is prohibited and may be subject to civil	*/
/*	and criminal penalties. Disclosure to others is			*/
/*	prohibited. For the terms and conditions of software use,	*/
/*	refer to the appropriate Digital Research Licence		*/
/*	Agreement.							*/
/*									*/
/************************************************************************/
/*									*/
/*	Sample NIOS for several popular ARCNET cards for IBM PCs	*/
/*									*/
/*	Written by:	JW, Digital Research EDC, Hungerford, GB	*/
/*									*/
/*	Date	   Who  Modification					*/
/*	---------  ---	-------------------------------------------	*/
/*	17-Nov-88  JW	Initially written, based on RASM source		*/
/*	 7-Dec-88  JW	check bufctrl, not stsin for deciding		*/
/*			whether to send or make a buffer pending	*/
/*	 8-Dec-88  JW	use 386 mapping, special AT PIC programming	*/
/*			accept good ACKs only				*/
/*	 9-Dec-88  JW	fix DL semaphore problem			*/
/*	10-Dec-88  JW	remove PIC reset race condition			*/
/*	12-Dec-88  JW	loop in hardware ISR while interrupts		*/
/*	23-Jan-89  JW	guarantee max. receiver disable period		*/
/*	 1-Jun-89  JW	scan both 2E0 and 220 for network I/O ports	*/
/*	 2-Jun-89  JW	remove 01F5h byte message size limit,		*/
/*			support deblocked logical messages		*/
/*	 5-Jun-89  JW	try to optimize flagsets & dispatches		*/
/*	14-Jun-89  JW	also check 'tseq' on continuation		*/
/*	15-Jun-89  JW	change data link semaphore code			*/
/*			use symbolic names for BDOS calls		*/
/*	23-Jun-89  JW	take deblocking loop out of DLOUT		*/
/*	27-Jun-89  JW	support multiple input line drivers		*/
/*	28-Jun-89  JW	force NETINs to free buffers after F75		*/
/*	27-Jul-89  JW	check log msg size on 1st partial vs. buf size	*/
/*	 4-Aug-89  JW	stop re-enable receive timer on input		*/
/*			remove 'dispatch', we now use DRL		*/
/*	 9-Aug-89  JW	support re-using init code & adapter space	*/
/*	11-Aug-89  JW	fix ack for discarded msg from 3rd node		*/
/*	29-Aug-89  JW	move PNID array into unused NIOSA data area	*/
/*									*/
/*	YET TO BE DONE: Map RAM at network board memory address on 386	*/
/*									*/
/************************************************************************/

#include "i:portab.h"
#include <dos.h>

#include "i:ccpm.h"

#define BOOL BOOLEAN			/* YES/NO values */

/* Assembly language code utilities */
/* -------------------------------- */

EXTERN VOID PASCAL memset (VOID *, BYTE, UWORD);
EXTERN VOID FAR * PASCAL MKPTR ();
EXTERN UWORD PASCAL __BDOS (BYTE, UWORD);
EXTERN UWORD PASCAL __SUPIF (BYTE, UWORD);
EXTERN VOID PASCAL farmove (VOID FAR *, VOID FAR *, UWORD);
EXTERN VOID PASCAL int_init (WORD);
EXTERN VOID PASCAL flagset (WORD);	/* set interrupt flag */
#ifdef V386
EXTERN VOID FAR * PASCAL v386_chk (VOID);
EXTERN UWORD PASCAL v386_ptr (VOID);
#endif
EXTERN UWORD PASCAL io_mem_alloc (UWORD, UWORD);

EXTERN VOID PASCAL insert_mfl (UWORD, UWORD);

VOID PASCAL init_end (VOID);		/* forward declaration */

GLOBAL VOID PASCAL xpt_int (VOID);	/* ACK not arrived in time */
GLOBAL VOID PASCAL dlt_int (VOID);	/* message not transmitted in time */
GLOBAL VOID PASCAL ses_int (VOID);	/* DOS Plus:  msg not rcvd in time */

EXTERN WORD dl_tmt;			/* DOS+ timeout value, NIOS header */

EXTERN struct
{
	UWORD resvd1;			/* transmit error count */
	UWORD retry_cnt [14];		/* retry counters */
	UWORD size_err_cnt;		/* received message too large */
	UWORD rx_err_cnt;		/* receive error counter */
	UWORD xp_err_cnt;		/* transport timeout counter */
	UWORD tmt_err_cnt;		/* dos plus session timeout counter */
	UWORD recon_cnt;		/* reconfiguration counter */
	UWORD tx_buff_cnt;		/* no tx buffers counter */
	UWORD rx_buff_cnt;		/* no rx buffers counter */
	UWORD dl_err_cnt;		/* datalink tx timeout counter */
	UWORD discard_cnt;		/* input message discarded counter */
	UWORD rcv_off_cnt;		/* rx disabled counter */
	UWORD por_cnt;			/* power on reset counter */
}	err;

typedef struct
{
	BYTE	node;			/* our node number */
	BYTE	dont_care[0x10];
	BYTE	passwd[8];		/* network password for LOGON */
}	PTBL;

PTBL FAR *parm;


typedef struct				/* NDOS data segment header */
{
	BYTE FAR *ndos_init;
	BYTE FAR *ndos_entry;
	BYTE FAR *ndos_data;
	BYTE FAR *nios_data;
	BYTE FAR *net_code;
	BYTE FAR *nios_code;
	PTBL FAR *parm_tbl;
	BYTE FAR *rct_ptr;
	BYTE FAR *stat_buf;
	BYTE FAR *dyn_buf;
}	HCB;

typedef struct		/* Requester Control Block Definition */
{
	BYTE FAR *rlink;	/* RCB link field */
	UWORD rid;		/* Requester ID (logical) */
	BYTE rsts;		/* RCB receive status */
	BYTE rsndsts;		/* RCB send status */
	BYTE rtype;		/* RCB type */
	BYTE rflg;		/* RCB flag number */
	LONG rtmo;		/* RCB timeout counter field */
	BYTE FAR *rmsg;		/* Message pointer */
	UWORD rmsiz;		/* Current message size */
	BYTE rldnum;		/* LDCB number */
}	RCB;

typedef struct
{
	WORD dummy;
}	RCT;

typedef struct			/* DR Net Logical Message Definition */
{
	BYTE fmt;		/* Format byte */
	WORD did;		/* Destination ID */
	WORD sid;		/* Source ID */
	BYTE fnc;		/* Network function code */
	WORD siz;		/* Application level message size */
	BYTE amsg[1];		/* Start of application message */
}	LMSG;

typedef struct			/* Transport Level Message Definition */
{
	BYTE tsys;
	BYTE tflg;		/* transport type flag */
	BYTE tdid;		/* logical node destination ID */
	BYTE tsid;		/* logical node source ID */
	BYTE tseq;		/* sequence number */
	BYTE tmsq;		/* multi-block sequence */

	BYTE tdata[1];		/* data for function */
}	TMSG;

				/* tflg values: */
#define	MO_OLD	1		/* Message xp msg type (compatible) */
#define	MAO	2		/* Message acknowledge type */
#define MO	3		/* Message output/input > 512 byte */
#define MNACK	4		/* negative acknowldgement */


#define	T_LEN_OLD 5		/* transport header length (excl. TMSQ) */
#define	T_LEN_TSQ 6		/* transport header length (incl. TMSQ) */
#define T_LEN T_LEN_TSQ		/* transport header length */

#define	FINAL 0x80		/* or in to mark as final block */

/* The following equates are specific to the ARCNET hardware */

#define	DRNET 253		/* system ID assigned by Datapoint */
				/*   for DR Net for use with ARCNET */

typedef struct			/* data link header for ARCNET */
{
	BYTE dl_sid;		/* physical source ID */
	BYTE dl_did;		/* physical destination ID */
	BYTE dl_cnt[2];		/* length of message */
	TMSG dl_tmsg;		/* actual message array */
}	DMSG;
#define	DL_LEN	4		/* size of everything before message */
#define MAX_TRY	10		/* max. # of attempts to deliver message */

DMSG dl_ack;			/* ACK message received into here */
DMSG dl_outack;			/* ACK message to send out */
#define	xp_ack dl_ack.dl_tmsg
#define	xp_outack dl_outack.dl_tmsg

#define ARCMAX_D 0x01FC		/* max msg via data link (4 byte overhead) */
#define ARCMAX_T 0x01F6		/* max msg via transport (10 byte overhead) */

BOOL dl_owned = NO;			/* data link semaphore */
BOOL dl_waiting = NO;


/*	Line driver header definition for NIOS	*/

typedef struct
{
	BYTE lnum;			/* Line driver number */
	BYTE ldid;			/* Destination ID */
	BYTE ltype;			/* Type--either input or output */
	UWORD bsiz;			/* Current buffer size */
	DMSG FAR *msg;			/* Message buffer pointer */
	BYTE FAR *init_f;		/* Initialization function pointer */
	BYTE FAR *drvr_f;		/* Driver function pointer */
	BYTE FAR *error_f;		/* Error function pointer */
}	LDCB;				/* Line Driver Control Block */

/*#define NUM_I_LDCB 2			/* # of input line drivers */
					/* Note: defined via -D option! */
#define NUM_O_LDCB 1			/* # of output line drivers */

enum
{
	DOSPLUS, CDOS, CDOS6
}	ostype = CDOS;

enum
{
	MAI_FLG,			/* ACK for message has arrived */
	REC_FLG,			/* message has arrived */
	TBE_FLG,			/* buffer has freed up */
	DLS_FLG,			/* data link semaphore freed up */
#if (NUM_I_LDCB > 1)
	XIN_FLG,			/* NETIN returned from XIN */
#endif
	SYN_FLG,			/* NETOUT done on early response */

	NUM_FLAGS
};

UWORD flags[NUM_FLAGS] =		/* flag numbers used */
{
	0x50, 0x51, 0x52, 0x53,		/* default #s for CDOS 5.x */
#if (NUM_I_LDCB > 1)
	0x54,
#endif
	0x55
};
BOOL waiting[NUM_FLAGS];		/* YES if we're waiting for this one */

#if (NUM_I_LDCB > 1)			/* if multiple input line drivers */
BOOL xin_owned = NO;			/* XIN semaphores */
BOOL xin_waiting = NO;
WORD xin_owner;				/* NETIN copy owning input */

BOOL xin_kludge[NUM_I_LDCB];		/* free up current buffer if YES */
#endif

#ifdef SYNC_RESPONSE
BOOL sync_wait = NO;			/* we're not waiting */
BYTE sync_fmt  = -1;			/* format not set yet */
WORD sync_did  = -1;			/* did not set yet */
#endif

/****** Specific hardware equates for our boards ******/

#define	INIT_FLGS	0x14            /* for pwron and recon flags */
#ifdef SMC
#define	XMT_BUF0	0x0000		/* first transmit buffer */
#define	XMT_BUF1	0x0200		/* second transmit buffer */
#define	RCV_BUF0	0x0400		/* first receive buffer */
#define	RCV_BUF1	0x0600		/* second receive buffer */

#define	INTWR		0x02e0		/* 9026 interrupt control */
#define	STSIN		0x02e0		/* 9026 chip status */
#define	CMDOUT		0x02e1		/* 9026 command register */

#define	ALT_FUDGE	(0x0220-0x02e0)

#define	DPARA		0x0400		/* scan on 16K boundaries */

#define	MPARA		0x0100		/* map 4 K of memory */
#endif

#ifdef DSP
#define	XMT_BUF0	0x0800		/* first transmit buffer */
#define	XMT_BUF1	0x0A00		/* second transmit buffer */
#define	RCV_BUF0	0x0C00		/* first receive buffer */
#define	RCV_BUF1	0x0E00		/* second receive buffer */

#define	INTWR		0x1800		/* 9026 interrupt control */
#define	STSIN		0x1800		/* 9026 chip status */
#define	CMDOUT		0x1801		/* 9026 command register */
#define	HARD_CLR	0x5000		/* un-reset DSP card */
#define	HARD_SET	0x7000		/* reset DSP card */

#define	DPARA		0x0800		/* scan on 32K boundaries */

#define	MPARA		0x0200		/* map 8 K of memory */
#endif

					/* Also see MYNIOSA for equates! */
#define INT_OFF		8		/* first hardware interrupt on IBM */

#define PC_LVL		2		/* PC: IRQ2 is spare - use it for ARCNET */
#define	AT_LVL		9		/* AT: use IRQ2/IRQ9 on slave */
#define	SLV_LVL		2		/* AT: slave PIC hooks into IRQ2 */
#define RED_LVL		9		/* AT: IRQ2 on bus is IRQ9 on slave */

#define TMR_LVL		0		/* we hook into timer - IRQ0 */
#define COM_EOI		(0x60|COM_LVL)	/* specific EOI to 8259A (Net H/W)*/

#define PIC_PORT	0x20		/* 8259A interrupt acknowledge port */
#define PIC_MASK	0x21		/* 8259A interrupt level mask */
#define	SLAVE_PORT	0xA0		/* 8259A slave ack port (AT) */
#define	SLAVE_MASK	0xA1		/* 8259A slave level mask (AT) */

UWORD irq = 0x00;			/* IRQ number for COM hardware */
BYTE pic_eoi, slv_eoi;			/* command for PIC */

UWORD bufmem = 0x0000;			/* buffer memory segment */
					/* 0 == auto scan */

#define	DXMIT		0x01		/* disable transmitter command */
#define	DRCV		0x02		/* disable receiver command */

#define	SND0	0x03			/* transmit from page 0 */
#define	SND1	0x0b			/* transmit from page 1 */
#define	SND2	0x13			/* transmit from page 2 */
#define	SND3	0x1b			/* transmit from page 3 */

#define	RCV0	0x04			/* receive from page 0 */
#define	RCV1	0x0c			/* receive from page 1 */
#define	RCV2	0x14			/* receive from page 2 */
#define	RCV3	0x1c			/* receive from page 3 */

#define	LNGCON	0x0d			/* set up for 512-byte packets */
#define	CLRFLG	0x1e			/* clear flags */

#define	RI	0x80			/* receiver inhibited */
#define	POR	0x10			/* power-on-reset */
#define	TEST	0x08			/* 9026 in test mode */
#define	RECON	0x04			/* reconfiguration occurred */
#define	TMA	0x02			/* transmit message acknowledged */
#define	TA	0x01			/* transmitter available */

#define BIDLE	0x00			/* buffer idle */
#define BPNDING	0x01			/* buffer waiting for a send */
#define BSNDING	0x02			/* buffer sending */
#define BRCVING	0x03			/* buffer receiving */
#define BRCVED	0x04			/* buffer with completed received message */

BYTE bufctrl[4] =			/* status of 4*512 byte RAM buffers */
{
	BIDLE,				/* Idle transmit buffer 0 */
	BIDLE,				/* Idle transmit buffer 1 */
	BRCVING,			/* Busy receive buffer 0 */
	BIDLE				/* Idle receive buffer 1 */
};

BYTE snd_cmd[4] =			/* send from buffer[n] */
{
	SND0, SND1, SND2, SND3
};

BYTE rcv_cmd[4] =			/* receive into buffer[n] */
{
	RCV0, RCV1, RCV2, RCV3
};

UWORD bufptrs[4] =			/* offsets of buffers */
{
	XMT_BUF0,			/* first transmit buffer */
	XMT_BUF1,			/* second transmit buffer */
	RCV_BUF0,			/* first receive buffer */
	RCV_BUF1			/* second receive buffer */
};

union
{
	struct
	{
	    UWORD buffo;
	    UWORD buffs;
	} w;
	BYTE FAR *buffp;
}	bufvar;
UWORD	bufadr;				/* physical network RAM segment */

#define	bufoff	bufvar.w.buffo		/* buffer offset */
#define	bufseg	bufvar.w.buffs		/* virtual buffer segment */
#define	bufptr	bufvar.buffp		/* virtual buffer address */

WORD nxtbuf = 2;			/* last buffer received */
BOOL rec_waiting = NO;			/* waiting for receiver flag */

#ifdef DSP
#define	stsin  STSIN			/* these values are constants */
#define	intwr  INTWR			/*   as they're memory offsets */
#define	cmdout CMDOUT			/*   on the DSP card segment */
#endif

#ifdef SMC
					/* The following port addresses are */
					/*   variable and ALT_FUDGE will be */
					/*   added to all of them if a card */
					/*   can't be found at the primary */
					/*   port address: */
UWORD stsin  = STSIN;			/* status port */
UWORD intwr  = INTWR;			/* enable/disable interrupt sources */
UWORD cmdout = CMDOUT;			/* enable/disable receiver/transmitter */
UWORD reset  = STSIN + 8;		/* OUT to RESET resets the chip */
#endif

BYTE int_mask = (RI /*|RECON*/ );	/* interrupt enable image */

/*
.pa
*/
/*	some timing stuff	*/

#define	TICKS_PER_SEC 60u		/* 60 Hz Tick on IBM PC XIOS */

/*	Conversion macros for delay()	*/

#define	MS(ms) ((ms*TICKS_PER_SEC+500)/1000)	/* round ms to ticks */
#define TICKS(t) t			/* ticks are normal quantity */

/*	NIOS internal timer control blocks	*/

typedef struct
{
	BOOL t_flag;			/* YES if timeout occurred */
	UWORD t_start;			/* programmed timeout */
	UWORD t_count;			/* remaining count, 0 == off */
}	TMR;				/* timer structure */


/*	Timing constants for NIOS		*/

#define XP_VAL	MS(2700) /* 670 */	/* transport output timeout */
					/* ACK must arrive by then */
#define DL_VAL	MS(335)			/* data link output timeout */
					/* chip must send message by then */
#define RX_VAL	MS(270)			/* receiver disable timeout */
					/* max. period receiver can be off */
#define TX_VAL	MS(50)			/* transmitter disabled timeout */
					/* max. period till transmitter idle */

#define START_TMR(t) {t.t_count=t.t_start;t.t_flag=NO;}
#define STOP_TMR(t) {t.t_count=0;t.t_flag=NO;}

TMR xpt_tmr = { NO, XP_VAL, 0 };	/* transport timeout */
TMR dlt_tmr = { NO, DL_VAL, 0 };	/* data link timeout */
TMR rxd_tmr = { NO, RX_VAL, 0 };	/* receiver disable timeout */
TMR tbe_tmr = { NO, TX_VAL, 0 };	/* transmitter disable timeout */
#ifdef DOSPLUS
TMR ses_tmr = { NO, 0,      0 };	/* session timer for DOS Plus only */
#endif

BOOL tbe_waiting = NO;			/* not waiting for buffer space */

/*	NIOS messages for signon		*/

BYTE cpyrite[] =
{
	"Copyright (C) 1984-89, Digital Research Inc."
};

EXTERN BYTE nocard_msg[];		/* "card not found" */
EXTERN BYTE noflag_msg[];		/* "out of flags" */

EXTERN BYTE signon_msg[];		/* "EDC NIOS VERS x.yy : " */
EXTERN BYTE hwinfo_msg[];		/* "Node=xx RAM=xxxx I/O=xxx etc" */

BYTE hextbl[] = "0123456789ABCDEF";	/* for printf("%x") hex output */

EXTERN BYTE *node_buf = (BYTE *) 0x0080;	/* keep this in startup code */
BYTE node_fcb [36] =
{
	0,
	'N','I','O','S','C','O','N','F',
	'S','Y','S',
	0, 0, 0, 0
};

WORD	afu = NO;			/* YES if hardware not present */


/*	Node table & equates for ARCNET		*/

#define	N_NODES	255
BYTE sseq[N_NODES];			/* send sequence number */
BYTE rseq[N_NODES];			/* receive sequence number */
EXTERN BYTE pnid[N_NODES];		/* physical node ID */

BOOL expect_ack = NO;			/* we don't expect ACK yet */
WORD expect_node;
BYTE expect_seq;			/* expected sequence # */
BYTE expect_id;				/* expected node # */

#define	PWRUP_CODE	0xD1		/* this is in RAM after reset */

#ifdef SMC				/* I/O port mapped I/O */
#define IO_DELAY io_delay();		/* call dummy function for delay */
#define	_IN_(p)		inp(p)		/* define port input */
#define	_OUT_(p,d)	outp(p,d)	/* define port output */
#endif
#ifdef DSP				/* memory mapped I/O */
					/* Note: shared pointer assumes: */
					/*	 I/O performed in */
					/*	 non-reentrant ISRs or */
					/*	 in CLI-ed regions only */
#define IO_DELAY			/* I/O delay not required for DSP */
MLOCAL
BYTE PASCAL _IN_ (UWORD port)		/* read from DSP port address */
{
	bufoff = port;
	return (*bufptr);
}

#define _OUT_(p,d) {bufoff=p; *bufptr=d;}
#endif

VOID PASCAL io_delay (VOID)		/* dummy function to waste time */
{
	/* calling us gives I/O ports time to recover */
}


GLOBAL
VOID PASCAL p_delay (WORD ticks)	/* delay calling process by >= ticks */
{
	
	__SUPIF (P_DELAY, ticks);	/* wait minimum # of ticks */
}

GLOBAL
VOID PASCAL dev_flagwait (WORD flag)	/* wait for specified flag */
{
	waiting[flag] = YES;		/* yes, we're waiting now */
	__SUPIF (DEV_WAITFLAG, flags[flag]);	/* wait on ACK/input flag */
	waiting[flag] = NO;		/* now clear it down */
}

GLOBAL
VOID PASCAL dev_flagset (WORD flag)	/* set specified flag */
{
	waiting[flag] = NO;		/* won't be waiting any more */
	flagset (flags[flag]);		/* wake up waiting process */
}


VOID PASCAL init_start (VOID) {};	/* dummy func = start of init code */


WORD PASCAL asc_hex (BYTE *cp, UWORD *uwp)
{
	WORD	i;			/* loop index for collecting digits */
	UWORD	n;			/* accumulated value */
	BYTE	c;			/* next character from string */

	n = 0;				/* initialize accumulator */
	for (i = 0; i<4; i++)		/* we read two-digit values */
	{
	    c = *cp++;			/* next digit from string */
	    if (c >= 'A' && c <= 'F')
		n = (n << 4) + (c - 'A' + 10);	/* is a hex digit */
	    else if (c >= '0' && c <= '9')
		n = (n << 4) + (c - '0');	/* is a decimal digit */
	    else if (i == 0)		/* need at least one digit */
		return (FAILURE);	/* not a digit, error! */
	    else
	        break;
	}
	*uwp = n;			/* return the result */
	return (SUCCESS);		/* this worked all right */
}

MLOCAL
VOID PASCAL hexnib (UWORD n)		/* print single nibble */
{
	__BDOS (C_WRITE, hextbl [n & 0x0f]);	/* print single character */
}

MLOCAL
VOID PASCAL hexbyt (UWORD h)
{
	hexnib (h >> 4);		/* print high nibble */
	hexnib (h & 0x0f);		/* print low nibble */
}

GLOBAL VOID CDECL printf (BYTE *, ...);	/* prototype for the following: */

GLOBAL
VOID CDECL printf (s, varargs)		/* old style definition (var args) */
BYTE *s;
UWORD varargs;
{
	UWORD *uwp;

	uwp = &varargs;			/* get first argument */
	while (*s)			/* while not end of string */
	{
	    if (*s != '%')		/* while not formatting info */
	    {
		__BDOS (C_WRITE, *s);	/* just print the stuff */
		s ++;			/* next character */
	    }
	    else
	    {
		while (*s++ != 'x')	/* ignore format info for now */
		    ;			/* skip format info */
		hexbyt (*uwp++);	/* print hexadecimal byte */
	    }
	}
}


#ifdef VIDEO_DEBUG
VOID PASCAL debchr (BYTE c, WORD col)
{
	((WORD FAR *) 0xB0000000) [col] = 0x7000 | c;
	((WORD FAR *) 0xB8000000) [col] = 0x7000 | c;
}


VOID PASCAL debnib (UWORD w, WORD col)
{
	debchr (hextbl[w & 0x0f], col);
}


VOID PASCAL debbyt (UWORD w, WORD col)
{
	debnib (w >> 4, col++);
	debnib (w & 0x0f, col);
}


VOID PASCAL debwrd (UWORD w, WORD col)
{
	debbyt (w >> 8, col);
	debbyt (w & 0x00ff, col+2);
}
#endif


#ifdef V386
VOID PASCAL mapwin (VOID)		/* reclaim adapter address space */
{
	LONG FAR *ptbl;
	LONG ladr;
	UWORD npara;
	UWORD buflen;

	if ((ostype == CDOS6) &&	/* only >= 2.0 support high memory */
	    ((ptbl = v386_chk ())!=0))	/* if we're running CDOS 386 */
	{
	    ptbl += (bufadr >> 8);	/* get relevant table entry */
	    buflen = 0x0000;		/* no paragraphs mapped yet */
	    npara = DPARA;		/* # of paragraphs required */
	    while ((npara != 0) &&
		   !(*ptbl & 0xFFF00000))
	    {
		if (!(npara & 0x03FF))	/* if new block required */
		{			/* find address & allocate */
		    ladr = (((LONG)v386_ptr ()) << 14);
		    if (!ladr)		/* out of extended memory */
			break;		/* stop mapping */
		}
		*ptbl = ladr | 0x07;	/* set one page table entry */
		ladr  += 0x00001000;	/* next physical address */
		ptbl ++;		/* next page table entry */
		npara  -= 0x0100;	/* count down # of paragraphs */
		buflen += 0x0100;	/* count mapped space */
	    }

	    if (buflen != 0)		/* if any RAM mapped */
	    {
		__emit__(0x0f, 0x22, 0xd8);	/* secret Flush TLB */
		insert_mfl (bufadr, buflen);	/* add space to XIOS_MFL */
	    }

	}
}
#endif


VOID PASCAL mapmem (UWORD offset)	/* offset to be referenced */
{
	LONG FAR *ptbl;
	LONG ladr;
	UWORD npara;

	bufoff = offset;		/* set buffer offset */
#ifdef V386
	if ((ostype == CDOS6) &&	/* only >= 2.0 support high memory */
	    ((ptbl = v386_chk ())!=0))	/* if we're running CDOS 386 */
	{				/* ptbl -> 386 page table for 1st MB */
	    if (!bufseg)		/* if no window yet, allocate it */
	        bufseg = io_mem_alloc (MPARA, 0x00FF);
					/* bufseg = allocated, aligned page */
	    npara = MPARA;		/* # of paragraphs required */
	    ptbl += (bufseg >> 8);	/* get relevant table entry */
	    ladr = ((((LONG)bufadr << 4) + (LONG)offset)
		    & 0xFFFFFF000)	/* clear A0..A11 */
		    | 0x000000007;	/* give us access to memory */
	    do
	    {
		*ptbl = ladr;		/* set one page table entry */
		ladr += 0x00001000;	/* next physical address */
		ptbl += 1;		/* next page table entry */
		npara -= 0x0100;	/* count down # of paragraphs */
	    } while (npara != 0);	/* until window mapped completely */

	    __emit__(0x0f, 0x22, 0xd8);	/* secret Flush TLB */

	    bufoff &= 0x0FFF;		/* we're within 1st mapped page */
	}
	else				/* straigh CDOS XM */
#endif
	    bufseg = bufadr;		/* simply set the pointer */
}

VOID PASCAL signon (VOID)
{
	printf (signon_msg, parm->node);	/* print signon message */
}

VOID PASCAL show_info (VOID)
{
	printf (hwinfo_msg,		/* print general blurb */
		bufadr>>8,		/* network memory address */
#ifdef SMC
		intwr>>4,		/* I/O port number */
#endif
		irq,			/* interrupt request line */
		flags[0],		/* first flag */
		flags[NUM_FLAGS-1]	/* last flag */
	       );
}

GLOBAL
VOID PASCAL fatal (BYTE *msg)
{
	signon ();			/* print signon message */
	printf (msg);			/* print the message */
	p_delay (MS(5000));		/* wait five seconds */
	afu = YES;			/* NIOS can't function */
}

GLOBAL					/* reset all sequence #s to 0000 */
VOID PASCAL clr_seqs (VOID)		/* clear sequence numbers */
{
	memset (rseq, 0x00, N_NODES);	/* reset all receiver sequences */
	memset (sseq, 0x00, N_NODES);	/* reset all sender sequences */
}

#ifdef DSP
GLOBAL
WORD PASCAL hard_reset (UWORD offset)	/* reset/unreset DSP card */
{
	MLOCAL BYTE dummy;
	for (bufadr = 0xC000; bufadr < 0xF000; bufadr += DPARA)
	{
	    mapmem (offset);		/* map physical memory if 386 */
	    disable ();
	    dummy = *bufptr;		/* (un)reset the card */
	    enable ();
	}
	p_delay (MS(350));		/* wait 350 ms */
}
#endif

GLOBAL
WORD PASCAL hard_init (BYTE pnode)	/* initialize network hardware */
{
	WORD i;
	BOOL pc_at;			/* are we running on PC/AT? */
	BYTE mask;

	bufseg = 0xF000;
	bufoff = 0xFFFE;		/* F000:FFFE -> model ID for IBM */
	pc_at = ((*bufptr == 0xFC) ||
		 (*bufptr == 0xF8));

	if (pc_at)			/* if we've got slave PIC */
	{
	    if (!irq)
		irq = AT_LVL;	    
	    if (irq == SLV_LVL)		/* redirect IRQ2 to IRQ9 on AT */
		irq = RED_LVL;		/* use default for AT on slave (IRQ9) */
	}
	else				/* XT, Model 30 */
	{
	    if (irq > 7)		/* if no slave, reject >= 8 */
		irq = 0;
	    if (!irq)
	        irq = PC_LVL;		/* use default IRQ level (IRQ2) */
	}

	if (irq > 7)			/* PC AT: Network on slave PIC */
	{
	    mask = inp (SLAVE_MASK) | (1<<(irq-8));
	    outp (SLAVE_MASK, mask);	/* shut down interrupts */
	    pic_eoi = 0x60 + SLV_LVL;
	    slv_eoi = 0x60 + (irq-8);
	}
	else				/* PC/XT/Model 30/ etc. */
	{
	    mask = inp (PIC_MASK) | (1<<irq);
	    outp (PIC_MASK, mask);	/* shut down interrupts */
	    pic_eoi = 0x60 + irq;
	}

	bufseg = 0x0000;		/* make sure 386 gets window */

	clr_seqs ();			/* reset sequence #s in node table */
	int_init ((irq <= 7) ?		/* setup timer/comms interrupt */
		  irq+0x08 : irq-8+0x70);

#ifdef SMC
	if (_IN_ (stsin) & TEST)	/* if primary port address doesn't work */
	{				/*    attempt to use alternate address */
	    stsin  += ALT_FUDGE;	/* status port */
	    intwr  += ALT_FUDGE;	/* interrupt mask port */
	    cmdout += ALT_FUDGE;	/* command output port */
	    reset  += ALT_FUDGE;	/* reset port */
	}

	_OUT_ (reset, 0);		/* reset chip */
	p_delay (MS(350));		/* wait 350 ms */
#endif

#ifdef DSP				/* find DSP ARCNET card */
	hard_reset (HARD_SET);		/* reset & wait 350 ms */
	hard_reset (HARD_CLR);		/* unreset & wait 350 ms */
#endif

	for (bufadr = 0xC000; bufadr < 0xF000; bufadr += DPARA)
	{
	    if (!bufmem || (bufmem == bufadr))	/* auto scan or right one */
	    {
		mapmem (XMT_BUF0);		/* set memory pointer */
		if (bufptr[0] == PWRUP_CODE)
		{
		    bufptr[10] = PWRUP_CODE;
		    if (bufptr[10] == PWRUP_CODE)
			break;
		}
	    }
	}
	if (bufadr == 0xF000)		/* card not found anywhere */
	    goto nocard;

	for (i=10; i!=0; i--)
	{
	    _OUT_ (cmdout, CLRFLG);	/* clear RECON & PWRUP flags */
	    IO_DELAY			/* wait one moment */
	    if (!(_IN_ (stsin) & INIT_FLGS))
		break;
	}

	if (!i)
	    goto nocard;		/* flags weren't cleared */

	bufoff = XMT_BUF0;
	if (!pnode)			/* enquire node from hardware */
	    pnode = bufptr[1];		/* get node from 2nd byte */

	disable ();			/* shut off interrupts */

	if (irq > 7)			/* PC AT: Network on slave PIC */
	{
	    outp (PIC_MASK, inp (PIC_MASK) & ~(1 << SLV_LVL));
	    outp (SLAVE_MASK, inp (SLAVE_MASK) & ~(1<<(irq-8)));
	}
	else				/* PC/XT/Model 30/ etc. */
	{
	    outp (PIC_MASK, inp (PIC_MASK) & ~(1<<irq));
	}

	_OUT_ (cmdout, LNGCON);
	IO_DELAY
	_OUT_ (cmdout, RCV2);		/* enable first receiver buffer */
	IO_DELAY
	_OUT_ (intwr, int_mask);	/* set initial interrupt mask */
	IO_DELAY
	enable ();

	return (pnode);			/* return node number */

nocard:
	fatal (nocard_msg);		/* "ARCNET board not found" */
	return (0xFF);			/* hardware not found */

}

UWORD PASCAL flag_alloc (VOID)		/* allocate system flag */
{
	return __BDOS (DEV_FLAGALLOC, 0);
}

BYTE to_upper (BYTE c)
{
	return ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

VOID PASCAL ninit (HCB FAR *hcb)	/* global NIOS initialization */
{					/* hcb -> NDOS data header */
	UWORD	init_b, init_e;

	UWORD	vers;
	UWORD	port;
	UWORD	mynode;
	WORD	i;
	BYTE	*cp;
	BYTE	c;

	vers = __BDOS (S_BDOSVER, 0);	/* get BDOS version */
	switch (vers >> 8)		/* get product code */
	{
#ifdef DOSPLUS
	 case 0x10:
	    ostype = DOSPLUS;
	    /* T.B.D. */		/* not currently supported */
	    break;
#endif

	 case 0x14:			/* this is CDOS 5.x or 6.x */
	 case 0x16:
	    ostype = CDOS;
	    if ((vers & 0x00FF) >= 0x60)
	    {
		ostype = CDOS6;
		for (i=0; i<NUM_FLAGS; i++)
		    flags[i] = flag_alloc ();	/* allocate flag */

		if (flags[NUM_FLAGS-1] == 0xFFFF)	/* if flag alloc err */
		{
		    fatal (noflag_msg);		/*   print error message */
		    return;
		}
	    }
	    break;
	}
	parm = hcb->parm_tbl;		/* get parameter table */

	mynode = N_NODES;		/* assume invalid value */

	__BDOS (F_DMAOFF, (UWORD)node_buf);	/* disk buffer = NODE_BUF */
	__BDOS (F_DMASEG, _DS);		/* DMA segment = local */
	__BDOS (F_ERRMODE, -1);		/* error mode = return */
	__BDOS (F_MULTISEC, 1);		/* sector count = 1 */

	if ((__BDOS (F_OPEN, (UWORD)(&node_fcb)) & 0xff) < 4)
	{
	    __BDOS (F_READ, (UWORD)(&node_fcb));	/* read one record */
	    __BDOS (F_CLOSE, (UWORD)(&node_fcb));	/* close node.### */
	    for (i=0; i<128; i++)
	    {
	        node_buf[i] = to_upper (node_buf[i]);
		if (node_buf[i] == 0x1a)
		    node_buf[i] = '\0';
	    }

	    cp = node_buf;
	    while ((c = *cp++) != '\0')
	    {
		while (*cp && (*cp++ != '='))
		    ;
		if (*cp == '\0')
		    break;
		switch (c)
		{
		 case 'N':			/* decode node number */
		    if ((asc_hex (cp, &mynode) == SUCCESS) &&
		        (mynode < N_NODES))
			/* go & use it */;
		    else
			mynode = N_NODES;
		    break;
#ifdef SMC
		 case 'P':
		    if ((asc_hex (cp, &port) == SUCCESS) &&
		        !(port & 0x000f))
		    {
			stsin = intwr = port;
			cmdout = port + 1;	/* command port = base+1 */
			reset  = port + 7;	/* reset port = base+8 */
		    }
		    break;
#endif
		 case 'M':
		    if ((asc_hex (cp, &bufmem) == SUCCESS) &&
		        !(bufmem & (DPARA-1)))	/* aligned memory address */
		    {
			/* only scan that address */
		    }
		    break;

		 case 'I':
		    if ((asc_hex (cp, &irq) == SUCCESS) &&
		        (irq >= 2) && (irq != 8) && (irq <= 15))
		    {
			/* use this IRQ # */
		    }
		    else
		        irq = 0;		/* else use default */
		    break;

		 default:
		    goto bad_opt;
		}

		while (*cp && (*cp++ != '\n'))		/* skip until CR/LFs */
		    ;
	    }
bad_opt:    ;

	}

	for (i=0; i<N_NODES; i++)	/* ARCNET: physical node #'s are */
	    pnid[i] = ~i;		/*         inverted logical nodes */

	i = hard_init (mynode < N_NODES ? pnid[mynode] : 0);	/* get node # */
	if (afu)			/* hardware not present */
	    return;

	for (mynode = 0; mynode < N_NODES; mynode++)
	{
	    if (pnid[mynode] == i)	/* look up node # in table */
	    {
		parm->node = mynode;
		break;
	    }
	}

	signon ();			/* display NIOS version */
	show_info ();			/* display NIOS/hardware info */
					/* now re-use init space */
#ifdef V386
	mapwin ();			/* re-use adapter address space */
#endif
					/* re-use initialization code as RAM */
	init_b = ((((UWORD) init_start) + 15) >> 4);
	init_e = (((UWORD) init_end) >> 4);
	insert_mfl (_CS + init_b, init_e - init_b);
}


VOID PASCAL init_end (VOID) {};		/* dummy func = start of init code */


VOID PASCAL dl_wait (VOID)		/* wait for dl out semaphore */
{
	disable ();			/* disable interrupts */
	while (dl_owned)		/* while we haven't got it */
	{
	    dl_waiting = YES;		/* indicate we're waiting for it */
	    enable ();			/* enable interrupts */
	    dev_flagwait (DLS_FLG);	/* wait for this flag */
	    disable ();			/* clear interrupts again */
	}
	dl_owned = YES;			/* not owned, let's get it ourselves */
	enable ();			/* re-enable interrupts */
}

VOID PASCAL dl_signal (VOID)		/* free up data-link semaphore */
{
	disable ();			/* disable interrupts */
	dl_owned = NO;			/* free up semaphore */
	if (!dl_waiting)		/* if nobody wants it */
	    enable ();			/* just re-enable interrupts */
	else				/* somebody flagwaiting */
	{
	    dl_waiting = NO;		/* reset waiting byte */
	    enable ();			/* re-enable interrupts */
	    dev_flagset (DLS_FLG);
	    __SUPIF (P_DISPATCH, 0);	/* force a dispatch */
	}
}

VOID PASCAL int_on (BYTE mask)		/* call this with ints disabled only */
{
	int_mask |= mask;		/* enable bit in image */
	_OUT_ (intwr, int_mask);	/* write to interrupt register */
}

WORD xmt_wait (VOID)			/* wait for transmit buffer space */
{
	WORD retry;			/* count attempts to get space */

	for (retry=10; retry != 0; retry--)
	{
	    disable ();
	    if (bufctrl[0] == BIDLE)	/* if 1st buffer not idle */
	    {
	        enable ();
		return (0);
	    }

	    if (bufctrl[1] == BIDLE)	/* if 2nd buffer idle */
	    {
	        enable ();
		return (1);
	    }

	    START_TMR (tbe_tmr);	/* start time-out */
	    tbe_waiting = YES;		/* get them to set the flag */
	    enable ();

	    dev_flagwait (TBE_FLG);	/* wait for transmitter available */
	}

	err.tx_buff_cnt ++;		/* tx buffer wait timed out */
	return (FAILURE);		/* return error code */
}

VOID PASCAL xmt_restart (WORD index)	/* restart transmitter */
{
	disable ();
	if (bufctrl[index^1] == BSNDING)	/* if transmitter not idle */
	{
	    bufctrl[index] = BPNDING;		/* mark as pending */
	}
	else					/* else transmitter available */
	{
	    bufctrl[index] = BSNDING;	/* mark buffer as being transmitted */
	    _OUT_ (cmdout, snd_cmd[index]);
	    START_TMR (dlt_tmr)		/* start data link timeout */
	}
	int_on (TA);			/* enable transmit available ints */
	enable ();			/* enable hardware interrupts */

}

UWORD PASCAL dlout (TMSG FAR *tmsg,	/* tmsg -> transport system header */
		    UWORD hsize,	/* size of hdr (usually T_LEN) */
		    BYTE FAR *src_ptr,	/* address of next partial message */
		    UWORD dsize)	/* count = # if bytes excl xp hdr */
{
	WORD index;			/* buffer index */
	UWORD msize;			/* complete block size */
	DMSG FAR *dmsg;			/* data link header on board RAM */
	BYTE FAR *dst_ptr;		/* address of message in board RAM */

	msize = hsize + dsize;		/* get physical message size */

	if ((index = xmt_wait ()) == FAILURE)
	    return (FAILURE);		/* couldn't find buffer space */
					/* else index = buffer to use */
	dmsg = MKPTR (bufptrs[index], bufseg);	/* get buffer pointer */

	if (msize <= (256-3))		/* small message sufficient */
	    dmsg->dl_cnt[0] = (-msize);	/* dl_cnt[0] = offset */
	else				/* large message, dl_cnt[0] = 0 */
	{				/* dl_cnt[1] = message offset */
	    if (msize < 257)		/* ARCNET can't handle medium size */
		msize = 257;		/*   messages - have to round up */
	    dmsg->dl_cnt[1] = (-msize);
	    dmsg->dl_cnt[0] = 0x00;	/* mark as long message */
	}

	dmsg->dl_did = pnid[tmsg->tdid];/* set physical node */
					/* source ID set by chip */
	dst_ptr = (BYTE FAR *)dmsg + ((WORD) ((UBYTE) (-msize)));
	farmove (tmsg, dst_ptr, hsize);	/* transport system message */
	if (dsize != 0)
	    farmove (src_ptr, dst_ptr+hsize, dsize);

	xmt_restart (index);		/* restart transmitter */

	return (SUCCESS);		/* message setup for transmission */
}					/*    but might not get there */

VOID PASCAL chk_log (LMSG FAR *lmsg, WORD node)
{
	if ((lmsg->fmt == 6) &&			/* Format 6 */
	    (lmsg->fnc == 64) &&		/* Function 64 */
	    (lmsg->amsg[20] == 0xFF))		/* first logon */
	{
	    sseq[node] = rseq[node] = 0;	/* reset sequence #s */
	}
}

VOID PASCAL send_ack (TMSG FAR *tmsg)
{
	WORD hsize;			/* variable header size */

	xp_outack.tflg = MAO;		/* got message, ACKnowledge it */
	xp_outack.tsys = DRNET;		/* mark as a DR NET message */
	xp_outack.tsid = tmsg->tdid;	/* swap sender vs. receiver */
	xp_outack.tdid = tmsg->tsid;
	xp_outack.tseq = tmsg->tseq;	/* echo with same sequence # */
	xp_outack.tmsq = tmsg->tmsq;

	hsize = (tmsg->tflg == MO_OLD) ?/* make ACK size compatible if */
		 T_LEN_OLD : T_LEN;	/*   request type was compatible */

	dl_wait ();			/* get data link semaphore */
	dlout (&xp_outack, hsize,	/* send ACK, minimum size */
	       (BYTE FAR *) 0, 0);
	dl_signal ();			/* release data link lock */
}

WORD PASCAL rcv_wait (VOID)			/* wait for received message */
{
	WORD index;				/* receive buffer index */

	STOP_TMR (rxd_tmr);			/* stop receiver timeout */

	disable ();				/* clear interrupts */
	FOREVER					/* while we're checking */
	{
	    index = nxtbuf^1;			/* assume its the other one */
	    if (bufctrl[index] == BRCVED)	/* look at other buffer 1st */
		break;				/* potentially the oldest */

	    index = nxtbuf;
	    if (bufctrl[index] == BRCVED)	/* look at most recent buffer */
		break;				/* either could have data */

	    rec_waiting = YES;			/* waiting for receiver */
	    enable ();				/* O.K. to have ints now */
#ifdef DOSPLUS
	    if (ostype == DOSPLUS)
	    {
		ses_tmr.t_start = dl_tmt;	/* get session timeout value */
		START_TMR (ses_tmr)		/* fire up session timer */
	    }
#endif
	    dev_flagwait (REC_FLG);		/* wait for receive flag */
#ifdef DOSPLUS
	    if (ses_tmr.t_flag)			/* if session timer timed out */
	    {
		err.tmt_err_cnt++;		/* one more session timeout error */
		return (FAILURE);		/* return session timeout */
	    }
#endif
	    disable ();				/* clear interrupts again */
	}					/* break if msg found */

	enable ();				/* admit interrupts again */
	return (index);

}

VOID PASCAL rcv_restart (WORD index)	/* restart receiver */
{
	disable ();			/* lock out receiver ints */
	STOP_TMR (rxd_tmr);		/* stop receiver timeout */
	bufctrl[index] = BIDLE;		/* mark received buffer as idle */
	index ^= 1;			/* select other buffer */
	if (bufctrl[index] != BRCVING)	/* if nobody receiving */
	{				/*   fire up receiver again */
	    index ^= 1;			/* select our own buffer */
	    bufctrl[index] = BRCVING;
	    _OUT_ (cmdout, rcv_cmd[index]);
	    int_on (RI);		/* enable receiver interrupt */
	}				/* else RI int will fire us up */
	enable ();			/* re-admit receiver ints */
}

UWORD PASCAL dlin (DMSG FAR *dmsg, UWORD icount)
					/* dmsg -> data link header */
					/* icount = buff size inc hdr + trl */
{
	TMSG FAR *tmsg;			/* transport system message */
	WORD index;			/* index of our message buffer */
	DMSG FAR *ndmsg;		/* DMSG in network board RAM */
	BYTE FAR *src_ptr;		/* source = msg in network RAM */
	LMSG FAR *dst_ptr;		/* destination = NDOS buffer */
	UWORD count;			/* remaining buffer size */
	UWORD total;			/* total message length */
	UWORD dsize;			/* LMSG logical size (inc lmsg hdr) */
	UWORD offset;			/* offset within buffer */
	WORD expmsq;			/* expected next sequence */
	TMSG xp_hdr;			/* local transport header */
	BYTE expnod;			/* expected source node */
	BYTE expseq;			/* expected sequence number */

	tmsg = &(dmsg->dl_tmsg);	/* get transport header */
	icount -= DL_LEN;		/* subtract header size */
	count = icount;			/* set remaining buffer size */
	expmsq = 0;			/* receive from beginning */
	do				/* receive blocks until tmsg complete */
	{
	    index = -1;			/* no need to restart originally */
again:	    if (index >= 0)
		rcv_restart (index);

	    if ((index = rcv_wait ()) == FAILURE)
		return (0xFFFE);	/* DOS Plus data link timeout */

	    ndmsg = MKPTR (bufptrs[index], bufseg);

	    total = 256;		/* assume short messages */
	    if ((offset = ndmsg->dl_cnt[0]) == 0)
	    {				/* if long message instead */
		total = 512;		/* total buffer size = 512 bytes */
		offset = ndmsg->dl_cnt[1];	/* get actual offset in buffer */
	    }

	    src_ptr = (BYTE FAR *) ndmsg + offset;
	    total -= offset;		/* # of bytes in physical message */
					/* now first grab transport header */
	    farmove (src_ptr, &xp_hdr, T_LEN);
	    if (xp_hdr.tflg == MO_OLD)	/* if old (compatible) message */
	    {				/*   we'll need to convert it */
		xp_hdr.tmsq = FINAL;	/* make up final message type */
		src_ptr += T_LEN_OLD;	/* skip old header */
		total   -= T_LEN_OLD;	/* we've moved some already */
	    }
	    else			/* this is the new protocol */
	    {
		src_ptr += T_LEN;	/* skip the message header */
		total   -= T_LEN;
	    }

	    if (!expmsq)		/* if this is the first one */
	    {
		if (xp_hdr.tmsq & 0x7F)	/* only allow initial messages */
		{
		    send_ack (&xp_hdr);
		    goto again;		/* reject subsequent ones */
		}
					/* any node will do initially: */
restart:
	        expnod = xp_hdr.tsid;	/* node whence this came from */
		expseq = xp_hdr.tseq;	/* save sequence number */
					/* save transport header */
	        farmove (&xp_hdr, tmsg, T_LEN);
	        count -= T_LEN;		/* adjust remaining space */
	    }
	    else			/* subsequent message in deblocking */
	    {
		if (xp_hdr.tsid != expnod)
		{
		    err.retry_cnt[12]++;
		    goto again;		/* throw away unexpected messages */
		}

		if (xp_hdr.tseq != expseq)
		{
		    send_ack (&xp_hdr);
		    goto again;		/* throw away unexpected messages */
		}

		if (!(xp_hdr.tmsq & 0x7F))
		{
		    count = icount;
		    expmsq = 0x00;
		    goto restart;
		}

		if ((xp_hdr.tmsq & 0x7F) != expmsq)
		{
		    send_ack (&xp_hdr);
		    goto again;
		}
	    }

	    dst_ptr = (LMSG FAR *) &(tmsg->tdata[(expmsq*ARCMAX_T)]);

	    if (total <= count)		/* if message will fit */
	    {
		farmove (src_ptr, dst_ptr, total);
		if (!expmsq)		/* if initial message */
		{
		    if ((dst_ptr->fmt < 6) ||
			(dst_ptr->fmt > 9))
		    {
			rcv_restart (index);	/* restart the receiver */
			return (FAILURE);
		    }

		    dsize = (dst_ptr->siz + 9);	/* get logical message size */
		    if (dsize+T_LEN > icount)	/* if larger than buffer */
			return (FAILURE);	/*   return error */
		}
	    }

	    rcv_restart (index);	/* restart the receiver */

	    if (total > count)		/* if message too long, return error */
	    {
		err.size_err_cnt++;	/* message size error */
		return (FAILURE);	/* discard the input */
	    }

	    count -= total;
	    expmsq ++;			/* expect next message */

	    tmsg->tmsq = xp_hdr.tmsq;	/* return sequence number */
	    if (!(xp_hdr.tmsq & FINAL))
		send_ack (&xp_hdr);	/* send ACK for partial message */

	}
	while (!(xp_hdr.tmsq & FINAL));	/* repeat until last block marker */
	return (SUCCESS);		/* we've received correctly */
}

#if (NUM_I_LDCB > 1)
VOID PASCAL xin_wait (LDCB *ldcb)	/* get semaphore (multiple NETINs) */
{
	disable ();			/* exclude the world */
	if (!xin_owned)			/* is XIN code being executed? */
	{
	    xin_owned = YES;		/* no, we can take it over */
	    enable ();			
	}
	else
	{
	    xin_waiting = YES;		/* now we're waiting for it */
	    enable ();			/* let them come in again */
	    dev_flagwait (XIN_FLG);	/* wait for them to finish */
	}
	xin_owner = ldcb->lnum;		/* set current owner */
}

VOID PASCAL xin_signal (LDCB *ldcb)	/* release semaphore (multiple NETINs) */
{
	disable ();			/* lock out the world */

	if (xin_waiting)		/* other NETIN waiting? */
	{
	    xin_waiting = NO;		/* not much longer, we're gonna */
	    dev_flagset (XIN_FLG);	/*   wake them up right now */
	}
	else				/* other NETIN not waiting */
	    xin_owned = NO;		/* just release it til they come */
	enable ();
}
#endif

UWORD PASCAL xin_init (VOID)		/* NETIN calls this for input ldvr */
{					/* don't let NETIN come up if bad */
	__SUPIF (P_PRIORITY, 180);	/* improve our priority a bit */
	return (afu ? FAILURE : SUCCESS);
}

UWORD PASCAL xin (LDCB *ldcb)
{
	WORD ret;			/* return code to caller */
	WORD ok;			/* message to be returned */
	DMSG FAR *dmsg;			/* data link message pointer */
	TMSG FAR *tmsg;			/* transport message pointer */
	LMSG FAR *lmsg;			/* logical message */
	WORD node;			/* node we've received from */

#if (NUM_I_LDCB > 1)			/* if potential reentrancy problem: */
	xin_wait (ldcb);		/* wait for xin semaphore */

	if (xin_kludge[xin_owner])	/* if message size changed */
	{				/* return once to get new buffer */
	    xin_kludge[xin_owner] = NO;	/*   of the right size */
	    return (FAILURE);		/*   if we were the other guy */
	}
#endif

	dmsg = ldcb->msg;		/* get data link message */
	tmsg = &(dmsg->dl_tmsg);	/* get transport header address */
	lmsg = (LMSG FAR *) (&tmsg->tdata[0]);	/* get logical message */

	FOREVER
	{
	    ret = dlin (dmsg, ldcb->bsiz);
	    if (ret == FAILURE)
		break;

#ifdef DOSPLUS
	    if (ret != SUCCESS)		/* if DOS+ timeout error */
		break;			/*    return to NDOS */
#endif

	    node = tmsg->tsid;		/* get source node number */
	    chk_log (lmsg, node);	/* check sequence number for node */
	    ok = YES;
#ifdef SEQNZ
	    if (tmsg->tseq == rseq[node] ||	/* if in sequence */
	        !tmsg->tseq || !rseq[node])
	    {
		do
		    rseq[node]++;		/* update for next time */
		while (!rseq[node]);
	    }
	    else
	        ok = NO;			/* bad message sequence */
#else
	    if (tmsg->tseq == rseq[node])	/* if in sequence */
	    {
		rseq[node]++;			/* update for next time */
	    }
	    else if ((BYTE)(tmsg->tseq+1) == rseq[node])/* if last one again */
	    {
		ok = NO;			/* message is duplicate */
	    }
	    else				/* if bad sequence number */
	    {
		rseq[node] = tmsg->tseq+1;	/* just restart sequence */
	    }
#endif

	    send_ack (tmsg);			/* send ACK for TMSG */

	    if (ok)				/* if successfully received */
		break;				/* break out of retries */

	    err.rx_err_cnt++;			/* count receive errs */
	}					/* retry until OK */

#if (NUM_I_LDCB > 1)
	xin_signal (ldcb);			/* give other NETINs access */
#endif

#ifdef SYNC_RESPONSE
	disable ();
	if (((sync_fmt == 6) ||
	     (sync_fmt == 8)) &&
	    (lmsg->fmt == (BYTE)(sync_fmt+1)) &&
	    (lmsg->sid == sync_did))
	{
	    sync_wait = YES;			/* can't return yet */
	    enable ();				/* interrupts on again */
	    dev_flagwait (SYN_FLG);		/* wait for sync flag */
	}
	else
	    enable ();				/* else just re-enable */
#endif

	return (ret);				/* return good status */

}

UWORD PASCAL xin_err (VOID)		/* input error recovery */
{
	return (SUCCESS);		/* not implemented */
}

UWORD PASCAL xout (LDCB *ldcb)
{
	UWORD slen;			/* save message length */
	WORD retry;
	TMSG FAR *tmsg;
	BYTE FAR *src_ptr;
	WORD node;
	UWORD offset;			/* offset of data after header */
	UWORD hsize;			/* transport header size */
	UWORD dsize;			/* actual msg size, excluding above */
	WORD ndmsgs;			/* number of partial messages */

	LMSG FAR *lmsg;
	WORD i;
	WORD ret;

	ret = SUCCESS;			/* assume we're gonna succeed */

	slen = ldcb->bsiz-DL_LEN;	/* set transport message size */
					/* we ignore NIOS headers */
	node = ldcb->ldid;
	tmsg = &(ldcb->msg->dl_tmsg);	/* get transport header */
	lmsg = (LMSG FAR *) (&tmsg->tdata[0]);	/* get logical message */

#ifdef SYNC_RESPONSE
	disable ();
	sync_fmt = lmsg->fmt;		/* save format of message */
	sync_did = lmsg->did;		/* save destination of message */
	enable ();
#endif

	chk_log (lmsg, node);		/* check if logon message */

	tmsg->tsys = DRNET;		/* set system ID format */

	slen -= T_LEN;			/* subtract transport header */
	if (slen <= ARCMAX_T)		/* maintain compatibility */
	{				/*   for small messages */
	    ndmsgs = 1;			/* just one data link message */
	    tmsg->tflg = MO_OLD;	/* backwards compatible message */
	    hsize = T_LEN_OLD;		/* exclude tmsq from header */
	}
	else				/* new format required */
	{
	    ndmsgs = (slen+ARCMAX_T-1)/ARCMAX_T;
	    tmsg->tflg = MO;
	    hsize = T_LEN;		/* standard transport header size */
	}

	expect_node = node;		/* save node we're expecting ACK from */
	expect_id  =
	tmsg->tdid = ldcb->ldid;	/* get destination */
	tmsg->tsid = parm->node;	/* get source */
	tmsg->tmsq = 0x00;
	expect_seq =
	tmsg->tseq = sseq[node];	/* get sequence number */

	src_ptr = tmsg->tdata;		/* data beyond transport header */
	do
	{
	    dsize = (slen > ARCMAX_T) ?
		     ARCMAX_T : slen;	/* size of message w/o xport hdr */

	    if (ndmsgs == 1)		/* if this is the final message */
		tmsg->tmsq |= FINAL;	/*    mark as the last one */

#if 0
  {
     static int col = 0;
     debbyt (lmsg->fnc, col);
     col += 2;
     debchr (':', col);
     col += 2;
  }
  if (*((BYTE FAR *) 0x00400017) & 0x10)
  {
     *((BYTE FAR *) 0x00400017) &= 0xEF;
     ret = FAILURE;
     goto done;
  }
#endif
	    for (retry=0; retry<MAX_TRY; retry++)
	    {
		dl_wait ();		/* get data link semaphore */
		START_TMR (xpt_tmr)	/* arm the transport system timer */
		expect_ack = YES;	/* we now expect ACKs */
		dlout (tmsg, hsize,	/* call data link output function */
		       src_ptr, dsize);
		dl_signal ();		/* release data link semaphore */

		dev_flagwait (MAI_FLG);	/* wait on message acknowledge flag */
		if (!xpt_tmr.t_flag &&	/* if no transport timeout */
		    (xp_ack.tseq == sseq[node]) &&	/* & sequence # OK */
		    (xp_ack.tmsq == tmsg->tmsq))	/* sub-sequence # OK */
		{				/* (sender seq# echoed in ACK) */
		    goto succeed;		/* exit retry 'for' loop */
		}
		else
		{
		    err.xp_err_cnt++;		/* one more transport system err */
		    if (retry < MAX_TRY)
			err.retry_cnt[retry]++;	/* one more err at retry level n */
		}
	    } /* end for (retry=...) */
	    ret = FAILURE;
	    goto done;

succeed:
	    tmsg->tmsq ++;		/* next transport sequence */
	    src_ptr += dsize;		/* move beyond this portion */
	    slen    -= dsize;		/* subtract new portion */
	    
	} while (--ndmsgs != 0);	/* until all partial messages sent */

#ifdef SEQNZ
	do
	    sseq[node]++;
	while (!sseq[node]);		/* never accept 0 as sequence # */
#else
	sseq[node] ++;			/* update sequence number */
#endif

#if (NUM_I_LDCB > 1)
	if ((lmsg->fmt == 7) &&			/* Format 7 */
	    (lmsg->fnc == 75))			/* Function 75 */
	{
	    for (i=0; i<NUM_I_LDCB; i++)	/* buffer size will change */
		xin_kludge[i] = YES;		/* force NETIN to free buf */
	}
#endif

done:

#ifdef SYNC_RESPONSE
	sync_fmt = (-1);		/* message not pending */
	if (sync_wait)			/* if NETIN waiting for sync flag */
	{				/*    then we must wake it up */
	    sync_wait = NO;		/* reset the flag for now */
	    dev_flagset (SYN_FLG);	/* wake up the process */
	}
#endif

	return (ret);			/* we've made our best attempt */
}

GLOBAL
VOID PASCAL tbe_signal (VOID)		/* signal end of xmt buf wait */
{
	if (tbe_waiting)		/* if waiting for buffer */
	{
	    STOP_TMR (tbe_tmr);		/* stop the timer */
	    tbe_waiting = NO;
	    dev_flagset (TBE_FLG);	/* wake up the process */
	}
}

GLOBAL
VOID PASCAL recon_int (VOID)		/* reconfiguration occurred on NET */
{
	err.recon_cnt++;		/* count reconfigurations */
	clr_seqs ();			/* clear media sequence numbers */
	_OUT_ (cmdout, CLRFLG);		/* clear reconfiguration flag */
	IO_DELAY			/* give hardware some time */
	_OUT_ (intwr, int_mask);	/* output new interrupt mask */
}

GLOBAL
VOID PASCAL por_int (VOID)		/* power on reset interrupt */
{
	err.por_cnt++;			/* count power on reset event */
	_OUT_ (cmdout, CLRFLG);		/* clear power-on-reset flag */
}

GLOBAL
VOID PASCAL ta_int (VOID)		/* transmitter become available */
{
	WORD index;			/* buffer index */

	tbe_signal ();			/* signal available space */
	if (bufctrl[0] == BSNDING)
	{
	    bufctrl[0] = BIDLE;		/* make this one idle */
	}
	else if (bufctrl[1] == BSNDING)
	{
	    bufctrl[1] = BIDLE;		/* make this one idle */
	}

	if (bufctrl[0] == BPNDING)
	{
	    index = 0;			/* send buffer 0 */
	}
	else if (bufctrl[1] == BPNDING)
	{
	    index = 1;			/* send buffer 1 */
	}
	else
	{				/* no buffer that needs sending */
	    int_mask &= ~TA;		/* disable transmitter ints */
	    STOP_TMR (dlt_tmr)		/* stop datalink transmitter timer */
	    goto ta_exit;		/* share exit code */
	}

	bufctrl[index] = BSNDING;	/* mark buffer as sending */
	_OUT_ (cmdout, snd_cmd[index]);	/* enable transmitter */
	START_TMR (dlt_tmr)		/* start timeout for TA int */
	int_mask |= TA;			/* enable transmitter */

ta_exit:
	_OUT_ (intwr, int_mask);	/* update interrupt mask */

}

GLOBAL
VOID PASCAL rcv_int (VOID)		/* receiver inhibited */
{
	WORD index;			/* buffer index */
	DMSG FAR *dmsg;			/* data link message */
	TMSG FAR *tmsg;
	UWORD total;
	UWORD offset;

	if (bufctrl[2] == BRCVING)	/* receiving message */
	{
	    index = 2;
	    dmsg = MKPTR (RCV_BUF0, bufseg);
	}
	else if (bufctrl[3] == BRCVING)
	{
	    index = 3;	
	    dmsg = MKPTR (RCV_BUF1, bufseg);
	}
	else				/* spurious interrupt */
	{
	    err.rx_buff_cnt++;		/* count spurious receiver ints */
	    return;
	}

	if ((offset = dmsg->dl_cnt[0]) == 0)	/* short/long message format */
	    offset = dmsg->dl_cnt[1];	/* get actual offset in buffer */

	tmsg = (TMSG FAR *) ((BYTE FAR *) dmsg + offset);
					/* tmsg->transport level message */
	if (tmsg->tsys == DRNET)	/* if this is a DR-NET message */
	{
	    if (tmsg->tflg == MAO)	/* is this an ACK package? */
	    {				/* hooray, last msg acknowledged */
		if (expect_ack &&	/* do we actually expect one? */
		    (tmsg->tsid == expect_id) &&
		    (tmsg->tseq == sseq[expect_node]))
		{			/* ignore if wrong sequence # */
		    xp_ack.tflg = tmsg->tflg;	/* save ACK for sequence */
		    xp_ack.tdid = tmsg->tdid;	/*   check elsewhere */
		    xp_ack.tsid = tmsg->tsid;
		    xp_ack.tseq = tmsg->tseq;
		    xp_ack.tmsq = (offset > (0x100-T_LEN)) ?
				   FINAL : tmsg->tmsq;
		        
		    expect_ack = NO;	/* message not expected any more */
		    STOP_TMR (xpt_tmr)	/* stop transport timer */
		    dev_flagset (MAI_FLG);	/* set "message acknowledged" flag */
		}
		else
		    err.retry_cnt[13]++;

		_OUT_ (cmdout, rcv_cmd[index]);
		int_mask |= RI;		/* enable receiver interrupt */
	    }
	    else			/* not an acknowledge, must be data */
	    {
		nxtbuf = index;		/* record last data received */
#ifdef DOSPLUS
		STOP_TMR (ses_tmr)	/* stop session timer */
#endif
		if (rec_waiting)	/* if s.o. waiting for flag */
		{
		    rec_waiting = NO;
		    dev_flagset (REC_FLG);	/* set "message received" flag */
		}

		bufctrl[index] = BRCVED;
		index ^= 1;		/* select other buffer */
		if (bufctrl[index] == BIDLE)	/* if other hasn't got msg */
		{
		    bufctrl[index] = BRCVING;	/* restart receiver */
		    _OUT_ (cmdout, rcv_cmd[index]);
		    int_mask |= RI;	/* enable receiver interrupt */
		}
		else			/* both buffers got data in them */
		{
		    err.rcv_off_cnt++;	/* count # of times rcv disabled */
		    int_mask &= ~RI;	/* disable receiver */
		    START_TMR (rxd_tmr);/* don't disable for too long */
		}
	    }
	    _OUT_ (intwr, int_mask);	/* enable/disable receiver interrupt */
	}
}					/* end of receiver interrupt */

GLOBAL
VOID PASCAL com_int (VOID)		/* 9026 hardware interrupt handler */
{
	WORD intstat;			/* temporary chip status */

	_OUT_ (intwr, 0);		/* reset the interrupt source */
	IO_DELAY			/* give hardware some time */
	intstat = _IN_ (stsin);		/* read the interrupt status */
	do
	{
	    if (intstat & POR)		/* did we have POR interrupt? */
		por_int ();		/* power-on-reset */
	    intstat &= int_mask;	/* ignore disabled ints */
	    if (intstat & RECON)	/* reconfiguration int? */
		recon_int ();		/* reconfiguration occurred */
	    if (intstat & TA)		/* transmitter become available */
		ta_int ();		/* transmit next buffer if any */
	    if (intstat & RI)		/* received data available */
		rcv_int ();		/* signal received data */
					/* there is no INTREQ from the board */
					/* as the mask register has been zeroed */
	    if (irq > 7)
		outp (SLAVE_PORT, slv_eoi);
	    outp (PIC_PORT, pic_eoi);	/* reset interrupt controller */
					/* if new int is pending, it will */
	    io_delay ();		/* be handled after releasing the mask */
	    intstat = _IN_ (stsin);	/* read the interrupt status */
	} while (intstat & (int_mask | POR));	/* loop while more ints */
	IO_DELAY			/* wait a little while */
	_OUT_ (intwr, int_mask);	/* enable interrupts again */
}

GLOBAL
VOID PASCAL xpt_int (VOID)
{
	xpt_tmr.t_flag = YES;		/* indicate timeout */
	expect_ack = NO;		/* don't expect ACK any more */
	dev_flagset (MAI_FLG);		/* set MAI flag (message ACK) */
}

GLOBAL
VOID PASCAL dlt_int (VOID)		/* data link timeout */
{
	WORD index;			/* temporary buffer index */

	err.dl_err_cnt++;		/* data link timeout error */
	_OUT_ (cmdout, DXMIT);		/* disable transmitter altogether */
	if (bufctrl[0] == BSNDING)
	    index = 0;			/* 1st buffer failed to transmit */
	else if (bufctrl[1] == BSNDING)
	    index = 1;			/* 2nd buffer failed to transmit */
	else
	    return;			/* no transmission in progress */

	bufctrl[index] = BIDLE;		/* mark buffer as idle */

	index ^= 1;			/* get the other buffer */
	if (bufctrl[index] == BPNDING)	/* transmit if it's pending */
	{
	    bufctrl[index] = BSNDING;		/* mark buffer as sending */
	    _OUT_ (cmdout, snd_cmd[index]);	/* enable transmitter */
	    START_TMR (dlt_tmr)			/* start timeout for xmit */
	    int_mask |= TA;			/* enable transmitter */
	    _OUT_ (intwr, int_mask);		/* update interrupt mask */
	}

	tbe_signal ();			/* signal available space */
}

GLOBAL
VOID PASCAL rxd_int (VOID)		/* receiver disable timeout */
{
	WORD index;			/* temporary buffer index */

	index = nxtbuf ^ 1;		/* get the other buffer */
	bufctrl[index] = BRCVING;	/* discard that buffer */
	_OUT_ (cmdout, rcv_cmd[index]);	/* re-receive it again */
	int_mask |= RI;			/* enable RI interrupts */
	_OUT_ (intwr, int_mask);	/* update interrupt mask */
	err.discard_cnt++;		/* count discarded message */
}

#ifdef DOSPLUS
GLOBAL
VOID PASCAL ses_int (VOID)		/* DOS Plus: session timeout */
{
	ses_tmr.t_flag = YES;		/* session timeout occurred */
	dev_flagset (REC_FLG);		/* wake up receiver (DOS Plus only) */
}
#endif

GLOBAL
VOID PASCAL tmr_int (VOID)		/* tick interrupt occurred */
{
	if (xpt_tmr.t_count)		/* if transport timer active */
	{
	    if (!(--(xpt_tmr.t_count)))	/* check for time-out condition */
		xpt_int ();		/* yes, handle time-out error */
	}
	if (dlt_tmr.t_count)
	{
	    if (!(--(dlt_tmr.t_count)))
		dlt_int ();
	}
#ifdef DOSPLUS
	if (ses_tmr.t_count)		/* DOS Plus session timeout */
	{
	    if (!(--(ses_tmr.t_count)))
		ses_int ();
	}
#endif
	if (rxd_tmr.t_count)		/* receiver disabled timeout */
	{
	    if (!(--(rxd_tmr.t_count)))
		rxd_int ();
	}
	if (tbe_tmr.t_count)		/* transmitter disabled timeout */
	{
	    if (!(--(tbe_tmr.t_count)))
		tbe_signal ();
	}
}
